<?php
/*
Plugin Name: ADV License Client
Description: License client for adverservice
Version: 9.0.1
Author:  Zephyr
*/

if (!defined('ABSPATH')) {
    exit;
}

define('ADV_LIC_PLUGIN_BASENAME', plugin_basename(__FILE__));
define('ADV_LIC_SERVER_URL', 'https://adverservice.com/activation/index.php');

// Register options
function adv_lic_register_settings() {
    add_option('adv_license_key', '');
    add_option('adv_license_status', 'unknown');
    add_option('adv_license_last_message', '');
    add_option('adv_license_last_check', 0);

    register_setting('adv_lic_options_group', 'adv_license_key');
}
add_action('admin_init', 'adv_lic_register_settings');

// On license key change → auto-activate
function adv_lic_on_license_change($old_value, $value) {
    if ($old_value !== $value) {
        adv_lic_contact_server('activate');
    }
}
add_action('update_option_adv_license_key', 'adv_lic_on_license_change', 10, 2);

// Menu
function adv_lic_menu() {
    add_options_page(
        'ADV License Settings',
        'ADV License',
        'manage_options',
        'adv-lic',
        'adv_lic_options_page'
    );
}
add_action('admin_menu', 'adv_lic_menu');

// Settings page
function adv_lic_options_page() {
    if (!current_user_can('manage_options')) {
        return;
    }

    $notice = '';

    // Manual check / activate button
    if (isset($_POST['adv_lic_manual_check']) && check_admin_referer('adv_lic_manual_check_action', 'adv_lic_manual_check_nonce')) {

        // Call license server
        adv_lic_contact_server('activate');

        // Read updated status + message
        $status = get_option('adv_license_status', 'unknown');
        $msg    = get_option('adv_license_last_message', '');

        $notice = 'License status updated: ' . $status . ($msg ? ' - ' . $msg : '');
    }

    // Always read current values (after possible update)
    $license_key  = get_option('adv_license_key', '');
    $status       = get_option('adv_license_status', 'unknown');
    $last_message = get_option('adv_license_last_message', '');
    $last_check   = (int) get_option('adv_license_last_check', 0);
    ?>
    <div class="wrap">
        <h1>ADV License Settings</h1>

        <?php if ($notice): ?>
            <div class="notice notice-success is-dismissible">
                <p><?php echo esc_html($notice); ?></p>
            </div>
        <?php endif; ?>

        <form method="post" action="options.php">
            <?php settings_fields('adv_lic_options_group'); ?>
            <?php do_settings_sections('adv_lic_options_group'); ?>

            <table class="form-table">
                <tr valign="top">
                    <th scope="row">License Key</th>
                    <td>
                        <input type="text"
                               name="adv_license_key"
                               value="<?php echo esc_attr($license_key); ?>"
                               style="width: 350px; direction:ltr;" />
                        <p class="description">
                            This must match the License Key created in the adverservice panel.
                        </p>
                    </td>
                </tr>
            </table>

            <?php submit_button('Save License'); ?>
        </form>

        <hr>

        <form method="post" style="margin-top: 20px;">
            <?php wp_nonce_field('adv_lic_manual_check_action', 'adv_lic_manual_check_nonce'); ?>
            <input type="hidden" name="adv_lic_manual_check" value="1" />
            <input type="submit" class="button button-primary" value="Activate / Check License">
        </form>

        <h2 style="margin-top:25px;">License Status</h2>
        <table class="form-table">
            <tr>
                <th>Status</th>
                <td><strong><?php echo esc_html($status); ?></strong></td>
            </tr>
            <tr>
                <th>Message</th>
                <td><?php echo $last_message ? esc_html($last_message) : '-'; ?></td>
            </tr>
            <tr>
                <th>Last Check</th>
                <td>
                    <?php
                    if ($last_check > 0) {
                        echo esc_html(date_i18n('Y/m/d H:i', $last_check));
                    } else {
                        echo '-';
                    }
                    ?>
                </td>
            </tr>
        </table>
    </div>
    <?php
}

// Call license server
function adv_lic_contact_server($action = 'check') {
    $license_key = get_option('adv_license_key');

    if (!$license_key) {
        update_option('adv_license_status', 'no_key');
        update_option('adv_license_last_message', 'No license key provided.');
        update_option('adv_license_last_check', time());
        return;
    }

    $home_url = home_url();
    $parsed   = wp_parse_url($home_url);
    $domain   = isset($parsed['host']) ? $parsed['host'] : ( $_SERVER['HTTP_HOST'] ?? '' );

    $response = wp_remote_post(ADV_LIC_SERVER_URL, [
        'timeout' => 15,
        'body'    => [
            'action'      => $action,
            'license_key' => $license_key,
            'domain'      => $domain,
        ],
    ]);

    if (is_wp_error($response)) {
        update_option('adv_license_status', 'connection_error');
        update_option('adv_license_last_message', 'Could not connect to license server.');
        update_option('adv_license_last_check', time());
        return;
    }

    $body = wp_remote_retrieve_body($response);
    $data = json_decode($body, true);

    if (!is_array($data) || empty($data['status'])) {
        update_option('adv_license_status', 'invalid_response');
        update_option('adv_license_last_message', 'Invalid response from license server.');
        update_option('adv_license_last_check', time());
        return;
    }

    update_option('adv_license_status', sanitize_text_field($data['status']));
    update_option('adv_license_last_message', sanitize_text_field($data['message'] ?? ''));
    update_option('adv_license_last_check', time());
}

// Daily cron check (optional)
function adv_lic_schedule_cron() {
    if (!wp_next_scheduled('adv_lic_cron_check_event')) {
        wp_schedule_event(time(), 'daily', 'adv_lic_cron_check_event');
    }
}
add_action('wp', 'adv_lic_schedule_cron');

add_action('adv_lic_cron_check_event', function() {
    adv_lic_contact_server('check');
});

// Admin notice if not ok
function adv_lic_admin_notice() {
    if (!current_user_can('manage_options')) {
        return;
    }

    $status = get_option('adv_license_status', 'unknown');
    if (in_array($status, ['blocked', 'inactive', 'no_key', 'connection_error', 'invalid_response', 'expired', 'domain_not_registered', 'max_domains_reached', 'not_found'], true)) {
        $msg = get_option('adv_license_last_message', '');
        ?>
        <div class="notice notice-error">
            <p><strong>ADV License:</strong> License status: <code><?php echo esc_html($status); ?></code></p>
            <?php if ($msg): ?>
                <p><?php echo esc_html($msg); ?></p>
            <?php endif; ?>
        </div>
        <?php
    }
}
add_action('admin_notices', 'adv_lic_admin_notice');
